/** @file pseries.cpp
 *
 *  Implementation of class for extended truncated power series and
 *  methods for series expansion. */

/*
 *  GiNaC Copyright (C) 1999-2003 Johannes Gutenberg University Mainz, Germany
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <iostream>
#include <stdexcept>

#include "pseries.h"
#include "add.h"
#include "inifcns.h" // for Order function
#include "lst.h"
#include "mul.h"
#include "power.h"
#include "relational.h"
#include "operators.h"
#include "symbol.h"
#include "print.h"
#include "archive.h"
#include "utils.h"

namespace GiNaC {

GINAC_IMPLEMENT_REGISTERED_CLASS(pseries, basic)


/*
 *  Default ctor, dtor, copy ctor, assignment operator and helpers
 */

pseries::pseries() : inherited(TINFO_pseries) { }

void pseries::copy(const pseries &other)
{
	inherited::copy(other);
	seq = other.seq;
	var = other.var;
	point = other.point;
}

DEFAULT_DESTROY(pseries)


/*
 *  Other ctors
 */

/** Construct pseries from a vector of coefficients and powers.
 *  expair.rest holds the coefficient, expair.coeff holds the power.
 *  The powers must be integers (positive or negative) and in ascending order;
 *  the last coefficient can be Order(_ex1) to represent a truncated,
 *  non-terminating series.
 *
 *  @param rel_  expansion variable and point (must hold a relational)
 *  @param ops_  vector of {coefficient, power} pairs (coefficient must not be zero)
 *  @return newly constructed pseries */
pseries::pseries(const ex &rel_, const epvector &ops_) : basic(TINFO_pseries), seq(ops_)
{
	GINAC_ASSERT(is_a<relational>(rel_));
	GINAC_ASSERT(is_a<symbol>(rel_.lhs()));
	point = rel_.rhs();
	var = rel_.lhs();
}


/*
 *  Archiving
 */

pseries::pseries(const archive_node &n, lst &sym_lst) : inherited(n, sym_lst)
{
	for (unsigned int i=0; true; ++i) {
		ex rest;
		ex coeff;
		if (n.find_ex("coeff", rest, sym_lst, i) && n.find_ex("power", coeff, sym_lst, i))
			seq.push_back(expair(rest, coeff));
		else
			break;
	}
	n.find_ex("var", var, sym_lst);
	n.find_ex("point", point, sym_lst);
}

void pseries::archive(archive_node &n) const
{
	inherited::archive(n);
	epvector::const_iterator i = seq.begin(), iend = seq.end();
	while (i != iend) {
		n.add_ex("coeff", i->rest);
		n.add_ex("power", i->coeff);
		++i;
	}
	n.add_ex("var", var);
	n.add_ex("point", point);
}

DEFAULT_UNARCHIVE(pseries)

//////////
// functions overriding virtual functions from base classes
//////////

void pseries::print(const print_context & c, unsigned level) const
{
	if (is_a<print_tree>(c)) {

		c.s << std::string(level, ' ') << class_name()
		    << std::hex << ", hash=0x" << hashvalue << ", flags=0x" << flags << std::dec
		    << std::endl;
		unsigned delta_indent = static_cast<const print_tree &>(c).delta_indent;
		size_t num = seq.size();
		for (size_t i=0; i<num; ++i) {
			seq[i].rest.print(c, level + delta_indent);
			seq[i].coeff.print(c, level + delta_indent);
			c.s << std::string(level + delta_indent, ' ') << "-----" << std::endl;
		}
		var.print(c, level + delta_indent);
		point.print(c, level + delta_indent);

	} else if (is_a<print_python_repr>(c)) {
		c.s << class_name() << "(relational(";
		var.print(c);
		c.s << ',';
		point.print(c);
		c.s << "),[";
		size_t num = seq.size();
		for (size_t i=0; i<num; ++i) {
			if (i)
				c.s << ',';
			c.s << '(';
			seq[i].rest.print(c);
			c.s << ',';
			seq[i].coeff.print(c);
			c.s << ')';
		}
		c.s << "])";
	} else {

		if (precedence() <= level)
			c.s << "(";
		
		std::string par_open = is_a<print_latex>(c) ? "{(" : "(";
		std::string par_close = is_a<print_latex>(c) ? ")}" : ")";
		
		// objects of type pseries must not have any zero entries, so the
		// trivial (zero) pseries needs a special treatment here:
		if (seq.empty())
			c.s << '0';
		epvector::const_iterator i = seq.begin(), end = seq.end();
		while (i != end) {
			// print a sign, if needed
			if (i != seq.begin())
				c.s << '+';
			if (!is_order_function(i->rest)) {
				// print 'rest', i.e. the expansion coefficient
				if (i->rest.info(info_flags::numeric) &&
					i->rest.info(info_flags::positive)) {
					i->rest.print(c);
				} else {
					c.s << par_open;
					i->rest.print(c);
					c.s << par_close;
				}
				// print 'coeff', something like (x-1)^42
				if (!i->coeff.is_zero()) {
					if (is_a<print_latex>(c))
						c.s << ' ';
					else
						c.s << '*';
					if (!point.is_zero()) {
						c.s << par_open;
						(var-point).print(c);
						c.s << par_close;
					} else
						var.print(c);
					if (i->coeff.compare(_ex1)) {
						if (is_a<print_python>(c))
							c.s << "**";
						else
							c.s << '^';
						if (i->coeff.info(info_flags::negative)) {
							c.s << par_open;
							i->coeff.print(c);
							c.s << par_close;
						} else {
							if (is_a<print_latex>(c)) {
								c.s << '{';
								i->coeff.print(c);
								c.s << '}';
							} else
								i->coeff.print(c);
						}
					}
				}
			} else
				Order(power(var-point,i->coeff)).print(c);
			++i;
		}

		if (precedence() <= level)
			c.s << ")";
	}
}

int pseries::compare_same_type(const basic & other) const
{
	GINAC_ASSERT(is_a<pseries>(other));
	const pseries &o = static_cast<const pseries &>(other);
	
	// first compare the lengths of the series...
	if (seq.size()>o.seq.size())
		return 1;
	if (seq.size()<o.seq.size())
		return -1;
	
	// ...then the expansion point...
	int cmpval = var.compare(o.var);
	if (cmpval)
		return cmpval;
	cmpval = point.compare(o.point);
	if (cmpval)
		return cmpval;
	
	// ...and if that failed the individual elements
	epvector::const_iterator it = seq.begin(), o_it = o.seq.begin();
	while (it!=seq.end() && o_it!=o.seq.end()) {
		cmpval = it->compare(*o_it);
		if (cmpval)
			return cmpval;
		++it;
		++o_it;
	}

	// so they are equal.
	return 0;
}

/** Return the number of operands including a possible order term. */
size_t pseries::nops(void) const
{
	return seq.size();
}

/** Return the ith term in the series when represented as a sum. */
ex pseries::op(size_t i) const
{
	if (i >= seq.size())
		throw (std::out_of_range("op() out of range"));

	return seq[i].rest * power(var - point, seq[i].coeff);
}

/** Return degree of highest power of the series.  This is usually the exponent
 *  of the Order term.  If s is not the expansion variable of the series, the
 *  series is examined termwise. */
int pseries::degree(const ex &s) const
{
	if (var.is_equal(s)) {
		// Return last exponent
		if (seq.size())
			return ex_to<numeric>((seq.end()-1)->coeff).to_int();
		else
			return 0;
	} else {
		epvector::const_iterator it = seq.begin(), itend = seq.end();
		if (it == itend)
			return 0;
		int max_pow = INT_MIN;
		while (it != itend) {
			int pow = it->rest.degree(s);
			if (pow > max_pow)
				max_pow = pow;
			++it;
		}
		return max_pow;
	}
}

/** Return degree of lowest power of the series.  This is usually the exponent
 *  of the leading term.  If s is not the expansion variable of the series, the
 *  series is examined termwise.  If s is the expansion variable but the
 *  expansion point is not zero the series is not expanded to find the degree.
 *  I.e.: (1-x) + (1-x)^2 + Order((1-x)^3) has ldegree(x) 1, not 0. */
int pseries::ldegree(const ex &s) const
{
	if (var.is_equal(s)) {
		// Return first exponent
		if (seq.size())
			return ex_to<numeric>((seq.begin())->coeff).to_int();
		else
			return 0;
	} else {
		epvector::const_iterator it = seq.begin(), itend = seq.end();
		if (it == itend)
			return 0;
		int min_pow = INT_MAX;
		while (it != itend) {
			int pow = it->rest.ldegree(s);
			if (pow < min_pow)
				min_pow = pow;
			++it;
		}
		return min_pow;
	}
}

/** Return coefficient of degree n in power series if s is the expansion
 *  variable.  If the expansion point is nonzero, by definition the n=1
 *  coefficient in s of a+b*(s-z)+c*(s-z)^2+Order((s-z)^3) is b (assuming
 *  the expansion took place in the s in the first place).
 *  If s is not the expansion variable, an attempt is made to convert the
 *  series to a polynomial and return the corresponding coefficient from
 *  there. */
ex pseries::coeff(const ex &s, int n) const
{
	if (var.is_equal(s)) {
		if (seq.empty())
			return _ex0;
		
		// Binary search in sequence for given power
		numeric looking_for = numeric(n);
		int lo = 0, hi = seq.size() - 1;
		while (lo <= hi) {
			int mid = (lo + hi) / 2;
			GINAC_ASSERT(is_exactly_a<numeric>(seq[mid].coeff));
			int cmp = ex_to<numeric>(seq[mid].coeff).compare(looking_for);
			switch (cmp) {
				case -1:
					lo = mid + 1;
					break;
				case 0:
					return seq[mid].rest;
				case 1:
					hi = mid - 1;
					break;
				default:
					throw(std::logic_error("pseries::coeff: compare() didn't return -1, 0 or 1"));
			}
		}
		return _ex0;
	} else
		return convert_to_poly().coeff(s, n);
}

/** Does nothing. */
ex pseries::collect(const ex &s, bool distributed) const
{
	return *this;
}

/** Perform coefficient-wise automatic term rewriting rules in this class. */
ex pseries::eval(int level) const
{
	if (level == 1)
		return this->hold();
	
	if (level == -max_recursion_level)
		throw (std::runtime_error("pseries::eval(): recursion limit exceeded"));
	
	// Construct a new series with evaluated coefficients
	epvector new_seq;
	new_seq.reserve(seq.size());
	epvector::const_iterator it = seq.begin(), itend = seq.end();
	while (it != itend) {
		new_seq.push_back(expair(it->rest.eval(level-1), it->coeff));
		++it;
	}
	return (new pseries(relational(var,point), new_seq))->setflag(status_flags::dynallocated | status_flags::evaluated);
}

/** Evaluate coefficients numerically. */
ex pseries::evalf(int level) const
{
	if (level == 1)
		return *this;
	
	if (level == -max_recursion_level)
		throw (std::runtime_error("pseries::evalf(): recursion limit exceeded"));
	
	// Construct a new series with evaluated coefficients
	epvector new_seq;
	new_seq.reserve(seq.size());
	epvector::const_iterator it = seq.begin(), itend = seq.end();
	while (it != itend) {
		new_seq.push_back(expair(it->rest.evalf(level-1), it->coeff));
		++it;
	}
	return (new pseries(relational(var,point), new_seq))->setflag(status_flags::dynallocated | status_flags::evaluated);
}

ex pseries::subs(const lst & ls, const lst & lr, unsigned options) const
{
	// If expansion variable is being substituted, convert the series to a
	// polynomial and do the substitution there because the result might
	// no longer be a power series
	if (ls.has(var))
		return convert_to_poly(true).subs(ls, lr, options);
	
	// Otherwise construct a new series with substituted coefficients and
	// expansion point
	epvector newseq;
	newseq.reserve(seq.size());
	epvector::const_iterator it = seq.begin(), itend = seq.end();
	while (it != itend) {
		newseq.push_back(expair(it->rest.subs(ls, lr, options), it->coeff));
		++it;
	}
	return (new pseries(relational(var,point.subs(ls, lr, options)), newseq))->setflag(status_flags::dynallocated);
}

/** Implementation of ex::expand() for a power series.  It expands all the
 *  terms individually and returns the resulting series as a new pseries. */
ex pseries::expand(unsigned options) const
{
	epvector newseq;
	epvector::const_iterator i = seq.begin(), end = seq.end();
	while (i != end) {
		ex restexp = i->rest.expand();
		if (!restexp.is_zero())
			newseq.push_back(expair(restexp, i->coeff));
		++i;
	}
	return (new pseries(relational(var,point), newseq))
	        ->setflag(status_flags::dynallocated | (options == 0 ? status_flags::expanded : 0));
}

/** Implementation of ex::diff() for a power series.
 *  @see ex::diff */
ex pseries::derivative(const symbol & s) const
{
	epvector new_seq;
	epvector::const_iterator it = seq.begin(), itend = seq.end();

	if (s == var) {
		
		// FIXME: coeff might depend on var
		while (it != itend) {
			if (is_order_function(it->rest)) {
				new_seq.push_back(expair(it->rest, it->coeff - 1));
			} else {
				ex c = it->rest * it->coeff;
				if (!c.is_zero())
					new_seq.push_back(expair(c, it->coeff - 1));
			}
			++it;
		}

	} else {

		while (it != itend) {
			if (is_order_function(it->rest)) {
				new_seq.push_back(*it);
			} else {
				ex c = it->rest.diff(s);
				if (!c.is_zero())
					new_seq.push_back(expair(c, it->coeff));
			}
			++it;
		}
	}

	return pseries(relational(var,point), new_seq);
}

ex pseries::convert_to_poly(bool no_order) const
{
	ex e;
	epvector::const_iterator it = seq.begin(), itend = seq.end();
	
	while (it != itend) {
		if (is_order_function(it->rest)) {
			if (!no_order)
				e += Order(power(var - point, it->coeff));
		} else
			e += it->rest * power(var - point, it->coeff);
		++it;
	}
	return e;
}

bool pseries::is_terminating(void) const
{
	return seq.empty() || !is_order_function((seq.end()-1)->rest);
}


/*
 *  Implementations of series expansion
 */

/** Default implementation of ex::series(). This performs Taylor expansion.
 *  @see ex::series */
ex basic::series(const relational & r, int order, unsigned options) const
{
	epvector seq;
	numeric fac = 1;
	ex deriv = *this;
	ex coeff = deriv.subs(r, subs_options::no_pattern);
	const symbol &s = ex_to<symbol>(r.lhs());
	
	if (!coeff.is_zero())
		seq.push_back(expair(coeff, _ex0));
	
	int n;
	for (n=1; n<order; ++n) {
		fac = fac.mul(n);
		// We need to test for zero in order to see if the series terminates.
		// The problem is that there is no such thing as a perfect test for
		// zero.  Expanding the term occasionally helps a little...
		deriv = deriv.diff(s).expand();
		if (deriv.is_zero())  // Series terminates
			return pseries(r, seq);

		coeff = deriv.subs(r, subs_options::no_pattern);
		if (!coeff.is_zero())
			seq.push_back(expair(fac.inverse() * coeff, n));
	}
	
	// Higher-order terms, if present
	deriv = deriv.diff(s);
	if (!deriv.expand().is_zero())
		seq.push_back(expair(Order(_ex1), n));
	return pseries(r, seq);
}


/** Implementation of ex::series() for symbols.
 *  @see ex::series */
ex symbol::series(const relational & r, int order, unsigned options) const
{
	epvector seq;
	const ex point = r.rhs();
	GINAC_ASSERT(is_a<symbol>(r.lhs()));

	if (this->is_equal_same_type(ex_to<symbol>(r.lhs()))) {
		if (order > 0 && !point.is_zero())
			seq.push_back(expair(point, _ex0));
		if (order > 1)
			seq.push_back(expair(_ex1, _ex1));
		else
			seq.push_back(expair(Order(_ex1), numeric(order)));
	} else
		seq.push_back(expair(*this, _ex0));
	return pseries(r, seq);
}


/** Add one series object to another, producing a pseries object that
 *  represents the sum.
 *
 *  @param other  pseries object to add with
 *  @return the sum as a pseries */
ex pseries::add_series(const pseries &other) const
{
	// Adding two series with different variables or expansion points
	// results in an empty (constant) series 
	if (!is_compatible_to(other)) {
		epvector nul;
		nul.push_back(expair(Order(_ex1), _ex0));
		return pseries(relational(var,point), nul);
	}
	
	// Series addition
	epvector new_seq;
	epvector::const_iterator a = seq.begin();
	epvector::const_iterator b = other.seq.begin();
	epvector::const_iterator a_end = seq.end();
	epvector::const_iterator b_end = other.seq.end();
	int pow_a = INT_MAX, pow_b = INT_MAX;
	for (;;) {
		// If a is empty, fill up with elements from b and stop
		if (a == a_end) {
			while (b != b_end) {
				new_seq.push_back(*b);
				++b;
			}
			break;
		} else
			pow_a = ex_to<numeric>((*a).coeff).to_int();
		
		// If b is empty, fill up with elements from a and stop
		if (b == b_end) {
			while (a != a_end) {
				new_seq.push_back(*a);
				++a;
			}
			break;
		} else
			pow_b = ex_to<numeric>((*b).coeff).to_int();
		
		// a and b are non-empty, compare powers
		if (pow_a < pow_b) {
			// a has lesser power, get coefficient from a
			new_seq.push_back(*a);
			if (is_order_function((*a).rest))
				break;
			++a;
		} else if (pow_b < pow_a) {
			// b has lesser power, get coefficient from b
			new_seq.push_back(*b);
			if (is_order_function((*b).rest))
				break;
			++b;
		} else {
			// Add coefficient of a and b
			if (is_order_function((*a).rest) || is_order_function((*b).rest)) {
				new_seq.push_back(expair(Order(_ex1), (*a).coeff));
				break;  // Order term ends the sequence
			} else {
				ex sum = (*a).rest + (*b).rest;
				if (!(sum.is_zero()))
					new_seq.push_back(expair(sum, numeric(pow_a)));
				++a;
				++b;
			}
		}
	}
	return pseries(relational(var,point), new_seq);
}


/** Implementation of ex::series() for sums. This performs series addition when
 *  adding pseries objects.
 *  @see ex::series */
ex add::series(const relational & r, int order, unsigned options) const
{
	ex acc; // Series accumulator
	
	// Get first term from overall_coeff
	acc = overall_coeff.series(r, order, options);
	
	// Add remaining terms
	epvector::const_iterator it = seq.begin();
	epvector::const_iterator itend = seq.end();
	for (; it!=itend; ++it) {
		ex op;
		if (is_exactly_a<pseries>(it->rest))
			op = it->rest;
		else
			op = it->rest.series(r, order, options);
		if (!it->coeff.is_equal(_ex1))
			op = ex_to<pseries>(op).mul_const(ex_to<numeric>(it->coeff));
		
		// Series addition
		acc = ex_to<pseries>(acc).add_series(ex_to<pseries>(op));
	}
	return acc;
}


/** Multiply a pseries object with a numeric constant, producing a pseries
 *  object that represents the product.
 *
 *  @param other  constant to multiply with
 *  @return the product as a pseries */
ex pseries::mul_const(const numeric &other) const
{
	epvector new_seq;
	new_seq.reserve(seq.size());
	
	epvector::const_iterator it = seq.begin(), itend = seq.end();
	while (it != itend) {
		if (!is_order_function(it->rest))
			new_seq.push_back(expair(it->rest * other, it->coeff));
		else
			new_seq.push_back(*it);
		++it;
	}
	return pseries(relational(var,point), new_seq);
}


/** Multiply one pseries object to another, producing a pseries object that
 *  represents the product.
 *
 *  @param other  pseries object to multiply with
 *  @return the product as a pseries */
ex pseries::mul_series(const pseries &other) const
{
	// Multiplying two series with different variables or expansion points
	// results in an empty (constant) series 
	if (!is_compatible_to(other)) {
		epvector nul;
		nul.push_back(expair(Order(_ex1), _ex0));
		return pseries(relational(var,point), nul);
	}
	
	// Series multiplication
	epvector new_seq;
	int a_max = degree(var);
	int b_max = other.degree(var);
	int a_min = ldegree(var);
	int b_min = other.ldegree(var);
	int cdeg_min = a_min + b_min;
	int cdeg_max = a_max + b_max;
	
	int higher_order_a = INT_MAX;
	int higher_order_b = INT_MAX;
	if (is_order_function(coeff(var, a_max)))
		higher_order_a = a_max + b_min;
	if (is_order_function(other.coeff(var, b_max)))
		higher_order_b = b_max + a_min;
	int higher_order_c = std::min(higher_order_a, higher_order_b);
	if (cdeg_max >= higher_order_c)
		cdeg_max = higher_order_c - 1;
	
	for (int cdeg=cdeg_min; cdeg<=cdeg_max; ++cdeg) {
		ex co = _ex0;
		// c(i)=a(0)b(i)+...+a(i)b(0)
		for (int i=a_min; cdeg-i>=b_min; ++i) {
			ex a_coeff = coeff(var, i);
			ex b_coeff = other.coeff(var, cdeg-i);
			if (!is_order_function(a_coeff) && !is_order_function(b_coeff))
				co += a_coeff * b_coeff;
		}
		if (!co.is_zero())
			new_seq.push_back(expair(co, numeric(cdeg)));
	}
	if (higher_order_c < INT_MAX)
		new_seq.push_back(expair(Order(_ex1), numeric(higher_order_c)));
	return pseries(relational(var, point), new_seq);
}


/** Implementation of ex::series() for product. This performs series
 *  multiplication when multiplying series.
 *  @see ex::series */
ex mul::series(const relational & r, int order, unsigned options) const
{
	pseries acc; // Series accumulator

	// Multiply with remaining terms
	const epvector::const_iterator itbeg = seq.begin();
	const epvector::const_iterator itend = seq.end();
	for (epvector::const_iterator it=itbeg; it!=itend; ++it) {
		ex op = recombine_pair_to_ex(*it).series(r, order, options);

		// Series multiplication
		if (it==itbeg)
			acc = ex_to<pseries>(op);
		else
			acc = ex_to<pseries>(acc.mul_series(ex_to<pseries>(op)));
	}
	return acc.mul_const(ex_to<numeric>(overall_coeff));
}


/** Compute the p-th power of a series.
 *
 *  @param p  power to compute
 *  @param deg  truncation order of series calculation */
ex pseries::power_const(const numeric &p, int deg) const
{
	// method:
	// (due to Leonhard Euler)
	// let A(x) be this series and for the time being let it start with a
	// constant (later we'll generalize):
	//     A(x) = a_0 + a_1*x + a_2*x^2 + ...
	// We want to compute
	//     C(x) = A(x)^p
	//     C(x) = c_0 + c_1*x + c_2*x^2 + ...
	// Taking the derivative on both sides and multiplying with A(x) one
	// immediately arrives at
	//     C'(x)*A(x) = p*C(x)*A'(x)
	// Multiplying this out and comparing coefficients we get the recurrence
	// formula
	//     c_i = (i*p*a_i*c_0 + ((i-1)*p-1)*a_{i-1}*c_1 + ...
	//                    ... + (p-(i-1))*a_1*c_{i-1})/(a_0*i)
	// which can easily be solved given the starting value c_0 = (a_0)^p.
	// For the more general case where the leading coefficient of A(x) is not
	// a constant, just consider A2(x) = A(x)*x^m, with some integer m and
	// repeat the above derivation.  The leading power of C2(x) = A2(x)^2 is
	// then of course x^(p*m) but the recurrence formula still holds.
	
	if (seq.empty()) {
		// as a special case, handle the empty (zero) series honoring the
		// usual power laws such as implemented in power::eval()
		if (p.real().is_zero())
			throw std::domain_error("pseries::power_const(): pow(0,I) is undefined");
		else if (p.real().is_negative())
			throw pole_error("pseries::power_const(): division by zero",1);
		else
			return *this;
	}
	
	const int ldeg = ldegree(var);
	if (!(p*ldeg).is_integer())
		throw std::runtime_error("pseries::power_const(): trying to assemble a Puiseux series");

	// O(x^n)^(-m) is undefined
	if (seq.size() == 1 && is_order_function(seq[0].rest) && p.real().is_negative())
		throw pole_error("pseries::power_const(): division by zero",1);
	
	// Compute coefficients of the powered series
	exvector co;
	co.reserve(deg);
	co.push_back(power(coeff(var, ldeg), p));
	bool all_sums_zero = true;
	for (int i=1; i<deg; ++i) {
		ex sum = _ex0;
		for (int j=1; j<=i; ++j) {
			ex c = coeff(var, j + ldeg);
			if (is_order_function(c)) {
				co.push_back(Order(_ex1));
				break;
			} else
				sum += (p * j - (i - j)) * co[i - j] * c;
		}
		if (!sum.is_zero())
			all_sums_zero = false;
		co.push_back(sum / coeff(var, ldeg) / i);
	}
	
	// Construct new series (of non-zero coefficients)
	epvector new_seq;
	bool higher_order = false;
	for (int i=0; i<deg; ++i) {
		if (!co[i].is_zero())
			new_seq.push_back(expair(co[i], p * ldeg + i));
		if (is_order_function(co[i])) {
			higher_order = true;
			break;
		}
	}
	if (!higher_order && !all_sums_zero)
		new_seq.push_back(expair(Order(_ex1), p * ldeg + deg));
	return pseries(relational(var,point), new_seq);
}


/** Return a new pseries object with the powers shifted by deg. */
pseries pseries::shift_exponents(int deg) const
{
	epvector newseq = seq;
	epvector::iterator i = newseq.begin(), end  = newseq.end();
	while (i != end) {
		i->coeff += deg;
		++i;
	}
	return pseries(relational(var, point), newseq);
}


/** Implementation of ex::series() for powers. This performs Laurent expansion
 *  of reciprocals of series at singularities.
 *  @see ex::series */
ex power::series(const relational & r, int order, unsigned options) const
{
	// If basis is already a series, just power it
	if (is_exactly_a<pseries>(basis))
		return ex_to<pseries>(basis).power_const(ex_to<numeric>(exponent), order);

	// Basis is not a series, may there be a singularity?
	bool must_expand_basis = false;
	try {
		basis.subs(r, subs_options::no_pattern);
	} catch (pole_error) {
		must_expand_basis = true;
	}
		
	// Is the expression of type something^(-int)?
	if (!must_expand_basis && !exponent.info(info_flags::negint))
		return basic::series(r, order, options);
		
	// Is the expression of type 0^something?
	if (!must_expand_basis && !basis.subs(r, subs_options::no_pattern).is_zero())
		return basic::series(r, order, options);

	// Singularity encountered, is the basis equal to (var - point)?
	if (basis.is_equal(r.lhs() - r.rhs())) {
		epvector new_seq;
		if (ex_to<numeric>(exponent).to_int() < order)
			new_seq.push_back(expair(_ex1, exponent));
		else
			new_seq.push_back(expair(Order(_ex1), exponent));
		return pseries(r, new_seq);
	}

	// No, expand basis into series
	ex e = basis.series(r, order, options);
	return ex_to<pseries>(e).power_const(ex_to<numeric>(exponent), order);
}


/** Re-expansion of a pseries object. */
ex pseries::series(const relational & r, int order, unsigned options) const
{
	const ex p = r.rhs();
	GINAC_ASSERT(is_a<symbol>(r.lhs()));
	const symbol &s = ex_to<symbol>(r.lhs());
	
	if (var.is_equal(s) && point.is_equal(p)) {
		if (order > degree(s))
			return *this;
		else {
			epvector new_seq;
			epvector::const_iterator it = seq.begin(), itend = seq.end();
			while (it != itend) {
				int o = ex_to<numeric>(it->coeff).to_int();
				if (o >= order) {
					new_seq.push_back(expair(Order(_ex1), o));
					break;
				}
				new_seq.push_back(*it);
				++it;
			}
			return pseries(r, new_seq);
		}
	} else
		return convert_to_poly().series(r, order, options);
}


/** Compute the truncated series expansion of an expression.
 *  This function returns an expression containing an object of class pseries 
 *  to represent the series. If the series does not terminate within the given
 *  truncation order, the last term of the series will be an order term.
 *
 *  @param r  expansion relation, lhs holds variable and rhs holds point
 *  @param order  truncation order of series calculations
 *  @param options  of class series_options
 *  @return an expression holding a pseries object */
ex ex::series(const ex & r, int order, unsigned options) const
{
	GINAC_ASSERT(bp!=0);
	ex e;
	relational rel_;
	
	if (is_a<relational>(r))
		rel_ = ex_to<relational>(r);
	else if (is_a<symbol>(r))
		rel_ = relational(r,_ex0);
	else
		throw (std::logic_error("ex::series(): expansion point has unknown type"));
	
	try {
		e = bp->series(rel_, order, options);
	} catch (std::exception &x) {
		throw (std::logic_error(std::string("unable to compute series (") + x.what() + ")"));
	}
	return e;
}

} // namespace GiNaC
